<?php

namespace App\Services\Filesystem\Upload;

use Exception;
use Illuminate\Support\Str;
use Illuminate\Http\UploadedFile;
use Illuminate\Support\Facades\Storage;
use App\Services\Filesystem\Abstract\AbstractUploadService;

class VideoUploadService extends AbstractUploadService
{
    /**
     * Default video extension used when saving videos.
     *
     * @var string
     */
    protected $videoDefaultExtension = 'mp4';

    /**
     * Temporary location (on the "local" disk) where uploaded videos
     * are stored before being attached to a post.
     *
     * @var string
     */
    protected $videoTemporaryLocation = 'tmp/videos';

    /**
     * Upload a video file from a local path to the configured storage disk
     * and return its info.
     *
     * @param  string  $videoFileLocalPath
     * @return array
     * @throws \Exception
     */
    public function upload(string $videoFileLocalPath): array
    {
        try {
            // Decide where to store the final video file
            $videoPath = $this->determineStoragePath($this->videoDefaultExtension);

            // Read file contents and put on the chosen disk
            $resultState = Storage::disk($this->storageDisk)->put(
                $videoPath,
                file_get_contents($videoFileLocalPath)
            );

            if (! $resultState) {
                throw new Exception("Video upload to disk ({$this->storageDisk}) failed.");
            }

            return [
                'disk'       => $this->storageDisk,
                'video_path' => $videoPath,
            ];
        } catch (Exception $e) {
            throw new Exception($e->getMessage(), 0, $e);
        }
    }

    /**
     * Save the uploaded video temporarily on the "local" disk.
     * We are NOT using FFmpeg here anymore, so duration is just 0.
     *
     * @param  \Illuminate\Http\UploadedFile  $videoFile
     * @return array
     * @throws \Exception
     */
    public function tempSaveLocally(UploadedFile $videoFile): array
    {
        try {
            // Store the uploaded file in storage/app/tmp/videos on the "local" disk
            $videoTempPath = Storage::disk('local')->putFile(
                $this->videoTemporaryLocation,
                $videoFile
            );

            if (! $videoTempPath) {
                throw new Exception('Video upload on disk (local) failed.');
            }

            // No FFmpeg = no real duration; we return 0
            $durationSeconds = 0;

            return [
                'disk'       => $this->storageDisk,
                'video_path' => $videoTempPath,   // path relative to "local" disk
                'duration'   => null,             // or '00:00' if your code expects a string
                'seconds'    => $durationSeconds,
            ];
        } catch (Exception $e) {
            throw new Exception($e->getMessage(), 0, $e);
        }
    }

    /**
     * Generate a temporary file path for a video file.
     *
     * @param  string|null  $videoExtension
     * @return string
     */
    public function generateVideoTemporaryFilePath(?string $videoExtension = null): string
    {
        $videoExtension = $videoExtension ?: $this->videoDefaultExtension;
        $uuid = Str::uuid()->toString();

        return "{$this->videoTemporaryLocation}/{$uuid}.{$videoExtension}";
    }
}
