import { IGif, IChannel, GifID } from '@giphy/js-types';

/**
 * If you want gifs or stickers
 */
type MediaType = 'stickers' | 'gifs' | 'text' | 'videos';
/**
 * Filters results by specified rating.
 */
type Rating = 'pg' | 'g' | 'y' | 'pg-13' | 'r';
/**
 * Sorting options
 */
type SortTypes = 'relevant' | 'recent';
interface TypeOption {
    type?: MediaType;
}
interface PaginationOptions {
    limit?: number;
    offset?: number;
}
interface CategoriesOptions extends PaginationOptions {
}
interface SubcategoriesOptions extends PaginationOptions {
}
interface RelatedOptions extends PaginationOptions {
    type?: 'gifs' | 'stickers' | 'videos';
    rating?: Rating;
}
interface TrendingOptions extends PaginationOptions, TypeOption {
    rating?: Rating;
}
interface RandomOptions extends PaginationOptions {
    type?: 'gifs' | 'stickers';
    tag?: string;
    rating?: Rating;
}
interface SearchOptions extends PaginationOptions, TypeOption {
    sort?: SortTypes;
    rating?: Rating;
    lang?: string;
    channel?: string;
    explore?: boolean;
}

interface ResultMeta {
    msg: string;
    response_id: string;
    status: number;
}
interface ResultPagination {
    count: number;
    total_count: number;
    offset: number;
}
interface Result {
    meta: ResultMeta;
    pagination: ResultPagination;
}
interface ErrorResult {
    message?: string;
    meta?: ResultMeta;
}
interface GifResult extends Result {
    data: IGif;
}
interface GifsResult extends Result {
    data: IGif[];
}
interface NonPaginatedGifsResult {
    data: IGif[];
    meta: ResultMeta;
}
interface ICategory {
    gif?: IGif;
    name: string;
    name_encoded: string;
    subcategories: ICategory[];
}
interface CategoriesResult extends Result {
    data: ICategory[];
}
interface ChannelsResult extends Result {
    data: IChannel[];
}

/**
 * @class GiphyFetch
 * @param {string} apiKey
 * @param {object} qsParams
 */
declare class GiphyFetch {
    constructor(apiKey: string, qsParams?: Record<string, string>);
    /**
     * @hidden
     */
    private apiKey;
    /**
     * @hidden
     */
    private qsParams;
    /**
     * @hidden
     */
    private getQS;
    /**
     * A list of categories
     *
     * @param {CategoriesOptions} [options]
     * @returns {Promise<CategoriesResult>}
     */
    categories(options?: CategoriesOptions): Promise<CategoriesResult>;
    /**
     * Get a single gif by a id
     * @param {string} id
     * @returns {Promise<GifsResult>}
     **/
    gif(id: string, options?: {
        internal?: boolean;
    }): Promise<GifResult>;
    /**
     *
     * @function
     * Get gifs by an array of ids
     * @param {string[]} ids
     *
     * @function
     * Get gifs by category and subcategory
     * @param {string} category
     * @param {string} subcategory
     * @returns {Promise<GifsResult>}
     **/
    gifs(ids: string[]): Promise<GifsResult>;
    gifs(category: string, subcategory: string): Promise<GifsResult>;
    emoji(options?: PaginationOptions): Promise<GifsResult>;
    /**
     * Returns a list of all the default emoji variations
     *
     * @param {PaginationOptions} options
     * @returns {Promise<GifsResult>}
     **/
    emojiDefaultVariations(options?: PaginationOptions): Promise<GifsResult>;
    /**
     * Returns a list of gifs representing all the variations for the emoji
     *
     * @param {string} id
     * @returns {Promise<NonPaginatedGifsResult>}
     **/
    emojiVariations(id: GifID): Promise<NonPaginatedGifsResult>;
    animate(text: string, options?: PaginationOptions): Promise<GifsResult>;
    /**
     * @param term: string The term you're searching for
     * @param options: SearchOptions
     * @returns {Promise<GifsResult>}
     **/
    search(term: string, options?: SearchOptions): Promise<GifsResult>;
    /**
     * Get a list of subcategories
     * @param {string} category
     * @param {SubcategoriesOptions} options
     * @returns {Promise<CategoriesResult>}
     */
    subcategories(category: string, options?: SubcategoriesOptions): Promise<CategoriesResult>;
    /**
     * Get trending gifs
     *
     * @param {TrendingOptions} options
     * @returns {Promise<GifsResult>}
     */
    trending(options?: TrendingOptions): Promise<GifsResult>;
    /**
     * Get a random gif
     * @param {RandomOptions} options
     * @returns {Promise<GifResult>}
     **/
    random(options?: RandomOptions): Promise<GifResult>;
    /**
     * Get related gifs by a id
     * @param {string} id
     * @param {SubcategoriesOptions} options
     * @returns {Promise<GifsResult>}
     **/
    related(id: string, options?: RelatedOptions): Promise<GifsResult>;
    /**
     * Search for channels based on a term
     * @param {string} term
     * @param options: SearchOptions
     * @returns {Promise<ChannelsResult>}
     **/
    channels(term: string, options?: SearchOptions): Promise<ChannelsResult>;
}

declare let serverUrl: any;
declare const setServerUrl: (url: string) => void;

declare class FetchError extends Error {
    statusText: string;
    status: number;
    url: string;
    constructor(message: string, url: string, status?: number, statusText?: string);
}
declare class GeoFetchError extends FetchError {
}

/**
 * @hidden
 */
declare const gifPaginator: (fetchGifs: (offset: number) => Promise<GifsResult>, initialGifs?: IGif[]) => () => Promise<IGif[]>;

type RequestOptions = {
    apiVersion?: number;
    noCache?: boolean;
    normalizer?: (a: any) => any;
};
declare function request(url: string, options?: RequestOptions): Promise<Result>;

export { CategoriesOptions, CategoriesResult, ChannelsResult, ErrorResult, FetchError, GeoFetchError, GifResult, GifsResult, GiphyFetch, ICategory, MediaType, NonPaginatedGifsResult, PaginationOptions, RandomOptions, Rating, RelatedOptions, Result, ResultMeta, ResultPagination, SearchOptions, SubcategoriesOptions, TrendingOptions, TypeOption, gifPaginator, request, serverUrl, setServerUrl };
